/*
"SpeedOmatic" Scale Speed Checker for Model Railway, v2.26.
For RP2040Zero.
RJM, 23June2025.
Timing of model train travel over a known distance with calculation and display of its Scale Speed.
*/

//Sketch version:
char ver[]={"2.26"};

#include <SPI.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SH110X.h>

#define SCREEN_WIDTH 128 // OLED display width, in pixels
#define SCREEN_HEIGHT 64 // OLED display height, in pixels
#define OLED_RESET     -1 // Reset pin # (or -1 if sharing Arduino reset pin)
#define SCREEN_ADDRESS 0x3C // See datasheet for Address; 0x3D for 128x64, 0x3C for 128x32

Adafruit_SH1106G display=Adafruit_SH1106G(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire1, OLED_RESET);

//Configuration pins
#define scalePin 10
#define unitsPin 11
#define showInfoPin 12
#define modePin 13

//I2C pins
#define SDApin 14
#define SCLpin 15

//Sensor input pins
#define startPin 29
#define stopPin 27

//Options pins
#define dispModePin 26


char numRay[4]; //4 place array to store converted numericals

int n=0; //counter for "for/next" instructions

int sepn=0; //track distance between start & stop sensors, mm.
int scale=0; //determined by setting of "Scale" jumper. LOW (Jumper ON) for "OO" gauge, 1:76 scale. HIGH (Jumper OFF) for "HO" gauge, 1:87 scale.

bool Level=LOW; //Sensor ACTIVE level. LOW (Jumper ON) for ACTIVE LOW. HIGH (Jumper OFF) for ACTIVE HIGH.

unsigned long Tstart=0; //generic variable to record time at start of ALL timing loops.
unsigned long Tstop=0; //time when "stop" sensor triggered
int scaleSpeed=0; //calculated & rounded scale speed as an integer
int timeOut=0; //time (in Seconds) to wait for a stop trigger before declaring false start and resetting
//NB: In setup(), the calculation of timeOut as a portion of the defined timeOutMax value overrides the timeOut value assigned here at variable declaration 

//define time delays
#define Tsplash 2000 //time to display splashscreen (mSec)
#define Tmodemsg 2000 //time to display "mode =" message (mSec)
#define Tconfig 4000 //time to display configuration info at startup (mSec)
#define Ttest 5000 //time sensors must be inactive before allowing a new speed test (mSec)
#define timeOutMax 32 //maximum false start timeout (Seconds)
#define Tspeed 5000 /time to display a detected speed value (mSec)

//declare variables used only in Demo mode
int x=0; //counter
bool y=0; //km/h or mph flag


void setup()
 {
  //change I2C pins from defaults
  Wire1.setSDA(SDApin);Wire1.setSCL(SCLpin);Wire1.begin(); //specify SDA, SCL pins other than the defaults, re-initiate Wire1

  Serial.begin(9600);

  display.begin(SCREEN_ADDRESS, true);

  // initialise OLED display
  display.clearDisplay();display.display(); //Clear the buffer - start with clean display RAM
  display.setTextColor(SH110X_WHITE); //set display for coloured text on black background
   
  //splashscreen
  //draw 2 large rectangles
  display.drawRect(0,10,126,54,SH110X_WHITE); //rectangle, start at 0,10 128 wide 54 high
  display.drawRect(2,12,122,50,SH110X_WHITE); //rectangle, start at 2,12 124 wide 50 high
  //put text inside the rectangle
  display.setTextSize(1);
  display.setCursor(25,24);display.println("Speed'O'Matic");
  display.setCursor(28,38);display.print("mk2");display.setCursor(71,38);display.print("v");display.print(ver);
  //update display from RAM
  display.display();
  
  //show splashscreen for a time, clear display, wait a while
  delay(Tsplash);display.clearDisplay();display.display();delay(200);
  
  //configure Separation and Configuration pins
  for(n=0;n<14;n++) {pinMode(n,INPUT_PULLUP);} //configure 14 programming pins as pulled-up inputs

  //read separation pins, determine separation distance
  for(n=0;n<10;n++)
   {if (digitalRead(n)==LOW) {bitSet(sepn,n);}else{bitClear(sepn,n);}}

  //read scale pin, determine scale 
  if(digitalRead(scalePin)==LOW) {scale=76;}else{scale=87;} //Jumper ON; "OO" gauge, 1:76 scale. Jumper OFF; "HO" gauge, 1:87scale

  //Calculate timeout time.
  timeOut=round(timeOutMax*(float(sepn)/1023));

   //Setup Options Pins
  pinMode(26,INPUT_PULLUP);pinMode(28,INPUT_PULLUP);

  //DISPLAY MODE SETTING
  //for multi-mode sketch only  --  show mode as set by modePin
   display.clearDisplay(); //clear display RAM
   display.setTextSize(1);
   if(digitalRead(modePin)==LOW)
     {display.setCursor(39,28);display.println("Run Mode");}
     else
    {display.setCursor(9,28);display.println("Demonstration Mode");}
   display.display(); //tfr RAM to display
   delay(Tmodemsg); //show message for defined time "Tmodemsg"
   display.clearDisplay();display.display();delay(200); //brief blank display before moving on
 
  //Display Information
  if(digitalRead(showInfoPin)==LOW)
  {
    display.clearDisplay(); //ensure display RAM is clear 
    display.setTextSize(1);
    if(digitalRead(modePin)==LOW) //"run" mode, so show full info
      {
        //1st info line to display RAM

        //2nd info line to display RAM

        //3rd info line to display RAM
        display.setCursor(0,16); display.print("Scale:    ");
        if(digitalRead(scalePin)==LOW) {display.print("OO (1:76)");}else{display.print("HO (1:87)");}

        //4th info line to display RAM
        display.setCursor(0,24); display.print("Units:    ");
        if(digitalRead(unitsPin)==LOW) {display.print("km/h");}else{display.print("mph");}

        //5th info line to display RAM
        itoa(sepn,numRay,10);
        display.setCursor(0,32); display.print("Track:    ");
        if(sepn<10){display.print(" ");}
        if(sepn<100){display.print(" ");}
        display.print(numRay);display.println(" mm");

        //6th info line to display RAM
        itoa(timeOut,numRay,10);
        display.setCursor(0,40); display.print("Timeout:  ");
        if(timeOut<10){display.print(" ");}
        display.print(numRay);display.println(" Sec");

        //7th info line to display RAM
        display.setCursor(0,48); display.print("Reset:    ");
        if(digitalRead(dispModePin)==LOW) {display.print("Manual");}else{display.print("Auto");}

      }
    display.display(); //update display from display RAM
    delay(Tconfig); //display configuration for defined time Tconfig
    display.clearDisplay();display.display();delay(200); //brief blank display before moving on
 
  }

  //Initialise "run" mode (only if "run" mode is selected - skipped for "demo" mode)
    display.clearDisplay();
    if(digitalRead(modePin)==LOW) //"run" mode is set
    {
      display.setCursor(55,56); display.setTextSize(1);display.println("Initialising"); display.display();
      pinMode((startPin),INPUT_PULLUP); pinMode((stopPin),INPUT_PULLUP); //configure sensor input pins
      startInitial: //label
      Tstart=millis();//record/reset start of initialisation check
      sensorTest: //label
      if(digitalRead(startPin)==LOW || digitalRead(stopPin)==LOW) {goto startInitial;} //if either sensor is active start timing again
      if ((millis()-Tstart) < Ttest) {goto sensorTest;}
      display.clearDisplay(); display.display();
    }

 }

void loop()
 {
   if(digitalRead(modePin)==LOW) {run();}else{demo();} //call demo or run functions according to setting of modePin
 }


void demo() //demonstration count-up/toggle/reset, various text sizes and positioning
 {
 demoStart:  
   itoa(x,numRay,10); //convert numerical x to ASCII string in array "numRay", specifying x is a number to base 10
   display.setTextSize(4);
   display.setCursor(0,15);

   //print leading spaces as required
   if (x<10) {display.print(" ");}
   if (x<100) {display.print(" ");}
   display.println(numRay);

   display.setTextSize(2);display.setCursor(80,29);
   if(y==0) {display.println("km/h");} else {display.println("mph");}

   display.display(); //tfr RAM to display

   delay(1000);
   
   x=x+1; if(x>20) {x=0; y=!y;} //reset count at >20 and toggle km/h mph flag
   display.clearDisplay();

   goto demoStart; //loop back to after "Demonstration Mode" message
   
 }


void run()
 {
   runStart:  
 
  display.setCursor(97,56); display.setTextSize(1);
  display.println("Ready"); display.display();
  
  startPinTest: //wait for "Start" sensor activation
  while(digitalRead(startPin)==!Level) //do nothing while startPin is not active

  Tstart=millis();
  display.clearDisplay();display.display();//clear "Ready" message
  delay(200); //brief blank display
  
  display.setCursor(91,56); display.setTextSize(1);
  display.println("Timing"); display.display();
  
  stopPinTest: //now wait for "Stop" sensor

  if(digitalRead(stopPin)==!Level)

    {
      //stopPin is inactive - check for false start timeout
      if((millis() - Tstart) < timeOut*1000) //timeOut is in Seconds, so multiply it by 1000 to make it milliSeconds
        {goto stopPinTest;}
        else
        {goto clearForReset;}
    }

   Tstop=millis(); //stop Pin is active, record Stop time
   
  display.clearDisplay();display.display(); //clear "Ready" message
  delay(200); //brief blank display

calculate:
    if ((digitalRead(unitsPin))==LOW)
      {scaleSpeed=round((sepn*scale*3.6)/(Tstop-Tstart));} //calculate and round off scale speed in km/h
      else
      {scaleSpeed=round((sepn*scale*3.6)/(1.61*(Tstop-Tstart)));} //calculate and round off scale speed in mph

//convert speed numerical to ASCII character string in array "numRay"
  itoa(scaleSpeed,numRay,10);
  display.setTextSize(4);display.setCursor(0,15); //Set text parameters & cursor position for Speed display

//write leading spaces into display RAM as required
   if (scaleSpeed<10) {display.print(" ");}
   if (scaleSpeed<100) {display.print(" ");}
//write speed into dislay RAM
   display.println(numRay);

//Set text parameters & cursor position for speed Units display, write speed Units into display RAM
   display.setTextSize(2);display.setCursor(80,29);
   if(y==0) {display.println("km/h");} else {display.println("mph");}
   display.display(); //update display from display RAM

//display for time
  if((digitalRead(dispModePin)==LOW))
   {delay(Tspeed);}
   else 
   {while(digitalRead(dispModePin)==HIGH) {};}

clearForReset:
   display.clearDisplay();display.display();delay(200); //brief blank display before moving on
 
msgReset:
  display.setCursor(73,56); display.setTextSize(1);
  display.println("Resetting"); display.display();
 
startReset:
  Tstart=millis();

testReset:
    if(digitalRead(startPin)==LOW || digitalRead(stopPin)==LOW) {goto startReset;} //if either sensor is active start timing again
    if ((millis()-Tstart) < Ttest) {goto testReset;}
    display.clearDisplay(); display.display();delay(200); //brief blank display before moving on
    goto runStart; //loop back & repeat the process indefinitely

}